/**
 * WP Image Optimizer Admin JS
 */
(function($) {
    'use strict';

    var $message = $('#wpio-message');

    function showMessage(text, isError) {
        $message
            .removeClass('hidden wpio-error wpio-success')
            .addClass(isError ? 'wpio-error' : 'wpio-success')
            .text(text);
    }

    function hideMessage() {
        $message.addClass('hidden');
    }

    function setLoading(isLoading) {
        $('.wpio-card').toggleClass('wpio-loading', isLoading);
    }

    function doActivation(apiKey, requireTerms) {
        var $termsCheckbox = $('#wpio-terms-checkbox');

        // Check terms if required
        if (requireTerms && $termsCheckbox.length > 0 && !$termsCheckbox.is(':checked')) {
            showMessage('Please accept the Terms and Privacy Policy to continue', true);
            return;
        }

        hideMessage();
        setLoading(true);

        var data = {
            action: 'wpio_activate',
            nonce: wpio.nonce,
            api_key: apiKey
        };

        // Pass existing site_id when upgrading (to preserve R2 cache)
        if (wpio.site_id) {
            data.existing_site_id = wpio.site_id;
        }

        $.ajax({
            url: wpio.ajax_url,
            type: 'POST',
            data: data,
            success: function(response) {
                if (response.success) {
                    showMessage(response.data.message, false);
                    // Reload page after short delay to show new state
                    setTimeout(function() {
                        location.reload();
                    }, 1500);
                } else {
                    showMessage(response.data.message || 'Activation failed', true);
                    setLoading(false);
                }
            },
            error: function() {
                showMessage('Connection error. Please try again.', true);
                setLoading(false);
            }
        });
    }

    // Toggle license form visibility
    $('#wpio-show-license-form').on('click', function(e) {
        e.preventDefault();
        $('#wpio-license-form').removeClass('hidden');
        $('.wpio-pro-link').addClass('hidden');
    });

    $('#wpio-hide-license-form').on('click', function(e) {
        e.preventDefault();
        $('#wpio-license-form').addClass('hidden');
        $('.wpio-pro-link').removeClass('hidden');
        hideMessage();
    });

    // Free activation (main button, empty api_key = auto-generate free key)
    $('#wpio-activate-btn').on('click', function() {
        doActivation('', true); // Empty = free activation
    });

    // Pro activation from initial setup (with license key)
    $('#wpio-activate-pro-btn').on('click', function() {
        var apiKey = $('#wpio-api-key').val().trim();

        if (!apiKey) {
            showMessage('Please enter your license key', true);
            return;
        }

        doActivation(apiKey, true);
    });

    // Enter key on license input
    $('#wpio-api-key').on('keypress', function(e) {
        if (e.which === 13) {
            e.preventDefault();
            $('#wpio-activate-pro-btn').click();
        }
    });

    // Upgrade button (for existing free users)
    $('#wpio-upgrade-btn').on('click', function() {
        var apiKey = $('#wpio-upgrade-key').val().trim();

        if (!apiKey) {
            showMessage('Please enter your license key', true);
            return;
        }

        doActivation(apiKey, false); // No terms check needed for upgrade
    });

    // Enter key on upgrade input
    $('#wpio-upgrade-key').on('keypress', function(e) {
        if (e.which === 13) {
            e.preventDefault();
            $('#wpio-upgrade-btn').click();
        }
    });

    // Purge cache button
    $('#wpio-purge-btn').on('click', function() {
        var $btn = $(this);

        if ($btn.prop('disabled')) {
            return;
        }

        if (!confirm('Purge all cached images? They will be re-optimized on next request.')) {
            return;
        }

        $btn.prop('disabled', true).text('Purging...');
        hideMessage();

        $.ajax({
            url: wpio.ajax_url,
            type: 'POST',
            data: {
                action: 'wpio_purge_cache',
                nonce: wpio.nonce
            },
            success: function(response) {
                if (response.success) {
                    showMessage(response.data.message, false);
                } else {
                    showMessage(response.data.message || 'Purge failed', true);
                }
                $btn.prop('disabled', false).text('Purge CDN Cache');
            },
            error: function() {
                showMessage('Connection error. Please try again.', true);
                $btn.prop('disabled', false).text('Purge CDN Cache');
            }
        });
    });

    // Format bytes to human readable
    function formatBytes(bytes) {
        if (bytes === 0) return '0 B';
        var k = 1024;
        var sizes = ['B', 'KB', 'MB', 'GB'];
        var i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(1)) + ' ' + sizes[i];
    }

    // Format number with commas
    function formatNumber(num) {
        return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ',');
    }

    // Load stats on page load
    function loadStats() {
        var $stats = $('#wpio-stats');
        if ($stats.length === 0) return;

        var $loading = $stats.find('.wpio-stats-loading');
        var $content = $stats.find('.wpio-stats-content');
        var $error = $stats.find('.wpio-stats-error');

        $.ajax({
            url: wpio.ajax_url,
            type: 'POST',
            data: {
                action: 'wpio_get_stats',
                nonce: wpio.nonce
            },
            success: function(response) {
                $loading.addClass('hidden');

                if (response.success && response.data) {
                    var data = response.data;
                    var images = data.total_images || 0;
                    var originalBytes = data.total_original_bytes || 0;
                    var avifBytes = data.total_optimized_bytes || 0;
                    var savingsBytes = data.total_savings_bytes || 0;
                    var savingsPercent = data.total_savings_percent || 0;
                    var avgOriginal = data.avg_original_bytes || 0;
                    var avgAvif = data.avg_optimized_bytes || 0;
                    var avgSavings = data.avg_savings_bytes || 0;
                    var tier = data.tier || 'paid';
                    var limit = data.limit || 0;

                    // Show images count with limit for free tier
                    if (tier === 'free' && limit > 0) {
                        $('#wpio-stat-images').text(formatNumber(images) + ' / ' + formatNumber(limit));
                        var usagePercent = Math.min(100, Math.round((images / limit) * 100));
                        $('#wpio-stat-images-sub').text(usagePercent + '% of free tier used');

                        // Add usage class for styling
                        var $imagesCard = $('#wpio-stat-images').closest('.wpio-stat-card');
                        $imagesCard.removeClass('wpio-usage-ok wpio-usage-warning wpio-usage-exceeded');
                        if (images >= limit) {
                            $imagesCard.addClass('wpio-usage-exceeded');
                        } else if (usagePercent >= 80) {
                            $imagesCard.addClass('wpio-usage-warning');
                        } else {
                            $imagesCard.addClass('wpio-usage-ok');
                        }
                    } else {
                        $('#wpio-stat-images').text(formatNumber(images));
                        $('#wpio-stat-images-sub').text('');
                    }

                    $('#wpio-stat-original').text(formatBytes(originalBytes));
                    $('#wpio-stat-original-avg').text(formatBytes(avgOriginal) + ' avg');
                    $('#wpio-stat-avif').text(formatBytes(avifBytes));
                    $('#wpio-stat-avif-avg').text(formatBytes(avgAvif) + ' avg');
                    $('#wpio-stat-percent').text(savingsPercent + '%');
                    $('#wpio-savings-fill').css('width', savingsPercent + '%');
                    $('#wpio-savings-detail').text(
                        formatBytes(savingsBytes) + ' reduced  •  ' + formatBytes(avgSavings) + ' per image'
                    );

                    $content.removeClass('hidden');
                } else {
                    $error.text('Unable to load stats').removeClass('hidden');
                }
            },
            error: function() {
                $loading.addClass('hidden');
                $error.text('Unable to load stats').removeClass('hidden');
            }
        });
    }

    // Load stats when page ready
    $(document).ready(function() {
        loadStats();
    });

})(jQuery);
