<?php
/**
 * Settings Page
 *
 * Ultra-simple admin settings with one-click activation.
 */

if (!defined('ABSPATH')) {
    exit;
}

class WP_Image_Optimizer_Settings {

    private const PAGE_SLUG = 'wpimg-optimizer';
    private const OPTION_API_KEY = 'wpio_api_key';
    private const OPTION_SITE_ID = 'wpio_site_id';
    private const OPTION_CDN_URL = 'wpio_cdn_url';
    private const OPTION_STATUS = 'wpio_status';
    private const OPTION_REGISTERED_DOMAIN = 'wpio_registered_domain';
    private const OPTION_CACHE_VERSION = 'wpio_cache_version';

    private const API_BASE = 'https://api.wpimg.io';

    public function init(): void {
        add_action('admin_menu', [$this, 'add_menu_page']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('wp_ajax_wpio_activate', [$this, 'ajax_activate']);
        add_action('wp_ajax_wpio_purge_cache', [$this, 'ajax_purge_cache']);
        add_action('wp_ajax_wpio_get_stats', [$this, 'ajax_get_stats']);
        add_action('admin_init', [$this, 'check_domain_mismatch']);
        add_action('admin_notices', [$this, 'show_status_notices']);
    }

    /**
     * Show admin notices for payment issues
     */
    public function show_status_notices(): void {
        $status = get_option(self::OPTION_STATUS, '');
        $api_key = get_option(self::OPTION_API_KEY, '');
        $settings_url = admin_url('options-general.php?page=wpimg-optimizer');

        // Show subscription status issues
        if (!empty($api_key) && in_array($status, ['past_due', 'expired', 'paused', 'service_discontinued'], true)) {
            if ($status === 'past_due') {
                ?>
                <div class="notice notice-warning">
                    <p>
                        <strong>Image Optimizer by wpimg.io:</strong> Payment issue detected.
                        Please <a href="https://wpimg.io/billing" target="_blank">update your payment method</a> to avoid service interruption.
                        Questions? <a href="mailto:support@wpimg.io">support@wpimg.io</a>
                    </p>
                </div>
                <?php
            } elseif ($status === 'expired' || $status === 'paused') {
                ?>
                <div class="notice notice-error">
                    <p>
                        <strong>Image Optimizer by wpimg.io:</strong> Subscription <?php echo esc_html($status); ?>. Image optimization is disabled.
                        <a href="https://wpimg.io/billing" target="_blank">Manage subscription</a> |
                        <a href="<?php echo esc_url($settings_url); ?>">Re-activate</a> |
                        <a href="mailto:support@wpimg.io">support@wpimg.io</a>
                    </p>
                </div>
                <?php
            } elseif ($status === 'service_discontinued') {
                ?>
                <div class="notice notice-error">
                    <p>
                        <strong>Image Optimizer by wpimg.io:</strong> This service has been discontinued. Image optimization has been automatically disabled to prevent broken images on your site.
                        We apologize for any inconvenience. Questions? <a href="mailto:support@wpimg.io">support@wpimg.io</a>
                    </p>
                </div>
                <?php
            }
            return; // Don't show usage warnings if there's a subscription issue
        }

        // Show free tier usage warnings
        $usage_info = get_transient('wpio_usage_info');
        if (!$usage_info || $usage_info['tier'] !== 'free') {
            return;
        }

        $used = $usage_info['used'];
        $limit = $usage_info['limit'];
        $usage_percent = $limit > 0 ? ($used / $limit) * 100 : 0;

        if ($used >= $limit) {
            ?>
            <div class="notice notice-error">
                <p>
                    <strong>Image Optimizer by wpimg.io:</strong> Free tier limit reached (<?php echo esc_html(number_format($used)); ?>/<?php echo esc_html(number_format($limit)); ?> images).
                    New images are no longer being optimized.
                    <a href="https://wpimg.io/pricing" target="_blank">Upgrade to continue</a> |
                    <a href="<?php echo esc_url($settings_url); ?>">View stats</a>
                </p>
            </div>
            <?php
        } elseif ($usage_percent >= 80) {
            ?>
            <div class="notice notice-warning">
                <p>
                    <strong>Image Optimizer by wpimg.io:</strong> Approaching free tier limit (<?php echo esc_html(number_format($used)); ?>/<?php echo esc_html(number_format($limit)); ?> images, <?php echo esc_html(round($usage_percent)); ?>% used).
                    <a href="https://wpimg.io/pricing" target="_blank">Upgrade for unlimited</a> |
                    <a href="<?php echo esc_url($settings_url); ?>">View stats</a>
                </p>
            </div>
            <?php
        }
    }

    public function add_menu_page(): void {
        add_options_page(
            'Image Optimizer',
            'Image Optimizer',
            'manage_options',
            self::PAGE_SLUG,
            [$this, 'render_page']
        );
    }

    public function enqueue_scripts($hook): void {
        if ($hook !== 'settings_page_' . self::PAGE_SLUG) {
            return;
        }

        wp_enqueue_style(
            'wpio-admin',
            plugins_url('../assets/admin.css', __FILE__),
            [],
            WPIO_VERSION
        );

        wp_enqueue_script(
            'wpio-admin',
            plugins_url('../assets/admin.js', __FILE__),
            ['jquery'],
            WPIO_VERSION,
            true
        );

        wp_localize_script('wpio-admin', 'wpio', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('wpio_nonce'),
            'site_url' => get_site_url(),
            'site_id' => get_option(self::OPTION_SITE_ID, ''),
        ]);
    }

    public function ajax_activate(): void {
        check_ajax_referer('wpio_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $api_key = sanitize_text_field($_POST['api_key'] ?? '');
        $existing_site_id = sanitize_text_field($_POST['existing_site_id'] ?? '');

        // Generate free key if none provided (free tier activation)
        $is_free_activation = empty($api_key);
        if ($is_free_activation) {
            $api_key = 'wpimgfree-' . wp_generate_uuid4();
        }

        $domain = get_site_url();

        // Development mode - store key locally, skip API call entirely
        if (self::is_dev_environment()) {
            update_option(self::OPTION_API_KEY, $api_key);
            update_option(self::OPTION_STATUS, 'development');
            update_option(self::OPTION_SITE_ID, '');
            update_option(self::OPTION_CDN_URL, '');
            update_option(self::OPTION_REGISTERED_DOMAIN, $domain);

            wp_send_json_success([
                'message' => 'Development mode detected. CDN disabled for local environment.',
                'status' => 'development',
            ]);
            return;
        }

        // Production mode - call API to activate
        $request_body = [
            'api_key' => $api_key,
            'domain' => $domain,
        ];

        // Include existing site_id when upgrading (to preserve R2 cache)
        if (!empty($existing_site_id)) {
            $request_body['existing_site_id'] = $existing_site_id;
        }

        $response = wp_remote_post(self::API_BASE . '/activate', [
            'timeout' => 15,
            'headers' => ['Content-Type' => 'application/json'],
            'body' => json_encode($request_body),
        ]);

        if (is_wp_error($response)) {
            wp_send_json_error(['message' => 'Connection failed: ' . $response->get_error_message()]);
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code !== 200 || empty($body['success'])) {
            $error = $body['error'] ?? 'Activation failed';
            wp_send_json_error(['message' => $error]);
        }

        update_option(self::OPTION_API_KEY, $api_key);
        update_option(self::OPTION_SITE_ID, $body['site_id']);
        update_option(self::OPTION_CDN_URL, $body['cdn_url']);
        update_option(self::OPTION_STATUS, $body['status']);
        update_option(self::OPTION_REGISTERED_DOMAIN, $domain);

        wp_send_json_success([
            'message' => 'Activated! Images are now being optimized.',
            'status' => $body['status'],
            'cdn_url' => $body['cdn_url'],
        ]);
    }

    public function ajax_purge_cache(): void {
        check_ajax_referer('wpio_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        // Increment cache version
        $current = (int) get_option(self::OPTION_CACHE_VERSION, 1);
        $new_version = $current + 1;
        update_option(self::OPTION_CACHE_VERSION, $new_version);

        wp_send_json_success([
            'message' => 'Cache purged. All images will be re-fetched from origin.',
            'version' => $new_version,
        ]);
    }

    public function ajax_get_stats(): void {
        check_ajax_referer('wpio_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $api_key = get_option(self::OPTION_API_KEY, '');
        if (empty($api_key)) {
            wp_send_json_error(['message' => 'Not activated']);
        }

        $response = wp_remote_post(self::API_BASE . '/stats', [
            'timeout' => 15,
            'headers' => ['Content-Type' => 'application/json'],
            'body' => json_encode(['api_key' => $api_key]),
        ]);

        if (is_wp_error($response)) {
            wp_send_json_error(['message' => 'Connection failed']);
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code !== 200 || empty($body['success'])) {
            wp_send_json_error(['message' => $body['error'] ?? 'Failed to fetch stats']);
        }

        // Store usage info for admin banner (expires in 1 hour)
        if (isset($body['tier']) && $body['tier'] === 'free' && isset($body['limit'])) {
            set_transient('wpio_usage_info', [
                'tier' => $body['tier'],
                'limit' => $body['limit'],
                'used' => $body['total_images'] ?? 0,
            ], HOUR_IN_SECONDS);
        } else {
            delete_transient('wpio_usage_info');
        }

        wp_send_json_success($body);
    }

    /**
     * Detect domain mismatch (site migration)
     */
    public function check_domain_mismatch(): void {
        $registered_domain = get_option(self::OPTION_REGISTERED_DOMAIN);
        $current_domain = get_site_url();
        $api_key = get_option(self::OPTION_API_KEY);

        // No API key set, nothing to do
        if (empty($api_key)) {
            return;
        }

        // Domain matches, nothing to do
        if ($registered_domain === $current_domain) {
            return;
        }

        // Domain changed - handle locally if dev, otherwise re-activate via API
        if (self::is_dev_environment()) {
            // Switched to dev environment - disable CDN locally
            update_option(self::OPTION_STATUS, 'development');
            update_option(self::OPTION_SITE_ID, '');
            update_option(self::OPTION_CDN_URL, '');
            update_option(self::OPTION_REGISTERED_DOMAIN, $current_domain);
            return;
        }

        // Production domain - trigger re-activation via API
        $this->background_reactivate($api_key, $current_domain);
    }

    /**
     * Re-activate with new domain (migration detected)
     */
    private function background_reactivate(string $api_key, string $domain): void {
        $response = wp_remote_post(self::API_BASE . '/activate', [
            'timeout' => 15,
            'headers' => ['Content-Type' => 'application/json'],
            'body' => json_encode([
                'api_key' => $api_key,
                'domain' => $domain,
            ]),
        ]);

        if (is_wp_error($response)) {
            return; // Silent fail, will retry next admin load
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code === 200 && !empty($body['success'])) {
            update_option(self::OPTION_SITE_ID, $body['site_id']);
            update_option(self::OPTION_CDN_URL, $body['cdn_url']);
            update_option(self::OPTION_STATUS, $body['status']);
            update_option(self::OPTION_REGISTERED_DOMAIN, $domain);
        }
    }

    /**
     * Check if current site is a development environment
     */
    private static function is_dev_environment(): bool {
        $domain = strtolower(get_site_url());
        return (
            strpos($domain, 'localhost') !== false ||
            strpos($domain, '127.0.0.1') !== false ||
            strpos($domain, '::1') !== false ||
            (bool) preg_match('/\.(local|test|ddev\.site|lndo\.site)/', $domain) ||
            strpos($domain, 'ngrok') !== false
        );
    }

    public function render_page(): void {
        if (!current_user_can('manage_options')) {
            return;
        }

        $api_key = get_option(self::OPTION_API_KEY, '');
        $site_id = get_option(self::OPTION_SITE_ID, '');
        $cdn_url = self::get_cdn_url();
        $status = get_option(self::OPTION_STATUS, '');
        $current_domain = get_site_url();
        $is_active = !empty($site_id) && $status !== 'development';
        $is_dev = $status === 'development';
        ?>
        <div class="wrap wpio-wrap">
            <h1>Image Optimizer by wpimg.io</h1>

            <?php if (empty($api_key)): ?>
                <!-- Not connected state - Free activation -->
                <div class="wpio-card wpio-card-connect">
                    <div class="wpio-card-body">
                        <h2 class="wpio-welcome-title">Optimize your images with one click</h2>

                        <p class="wpio-domain-info">
                            Site: <strong><?php echo esc_html($current_domain); ?></strong>
                        </p>

                        <div class="wpio-form-group wpio-checkbox-group">
                            <label class="wpio-checkbox-label">
                                <input type="checkbox" id="wpio-terms-checkbox">
                                <span>I agree to the <a href="https://wpimg.io/terms" target="_blank">Terms</a> and <a href="https://wpimg.io/privacy" target="_blank">Privacy Policy</a>. I understand that images will be optimized via external CDN infrastructure (e.g., Cloudflare).</span>
                            </label>
                        </div>

                        <button type="button" id="wpio-activate-btn" class="button button-primary button-hero">
                            Enable Image Optimization
                        </button>

                        <p class="wpio-pro-link">
                            Already have a Pro license? <a href="#" id="wpio-show-license-form">Enter license key</a>
                        </p>

                        <!-- Hidden license form for Pro users -->
                        <div id="wpio-license-form" class="wpio-license-form hidden">
                            <div class="wpio-form-group">
                                <label for="wpio-api-key">License Key</label>
                                <input type="text" id="wpio-api-key" class="wpio-input" placeholder="Enter your Pro license key">
                            </div>
                            <button type="button" id="wpio-activate-pro-btn" class="button button-primary">
                                Activate Pro
                            </button>
                            <a href="#" id="wpio-hide-license-form" class="wpio-cancel-link">Cancel</a>
                        </div>

                        <div id="wpio-message" class="wpio-message hidden"></div>
                    </div>
                </div>

            <?php elseif ($is_dev): ?>
                <!-- Development mode state -->
                <div class="wpio-card wpio-card-dev">
                    <div class="wpio-status-badge wpio-status-dev">Development Mode</div>
                    <div class="wpio-card-body">
                        <p>
                            Local development environment detected.<br>
                            Image optimization is <strong>disabled</strong> to prevent localhost URLs.
                        </p>
                        <p class="wpio-domain-info">
                            Current URL: <code><?php echo esc_html($current_domain); ?></code>
                        </p>
                        <p class="wpio-help-text">
                            Deploy to a public domain to enable optimization.
                        </p>

                        <div id="wpio-message" class="wpio-message hidden"></div>
                    </div>
                </div>

            <?php else:
                // Detect if this is a free tier user
                $is_free = strpos($api_key, 'wpimgfree-') === 0;
            ?>
                <!-- Connected state -->
                <div class="wpio-card wpio-card-active">
                    <div class="wpio-status-badge wpio-status-<?php echo $is_free ? 'free' : esc_attr($status); ?>">
                        <?php
                        if ($is_free) {
                            echo 'Active (Free)';
                        } else {
                            switch ($status) {
                                case 'trial':
                                    echo 'Trial Active';
                                    break;
                                case 'past_due':
                                    echo 'Payment Issue';
                                    break;
                                case 'paused':
                                    echo 'Paused';
                                    break;
                                case 'expired':
                                    echo 'Expired';
                                    break;
                                case 'service_discontinued':
                                    echo 'Service Discontinued';
                                    break;
                                default:
                                    echo 'Active (Pro)';
                            }
                        }
                        ?>
                    </div>
                    <div class="wpio-card-body">
                        <?php if ($status === 'past_due'): ?>
                            <div class="wpio-warning-banner">
                                <p><strong>Payment issue detected.</strong> Please update your payment method to avoid service interruption.</p>
                                <a href="https://wpimg.io/billing" target="_blank" class="button">Update Payment Method</a>
                                <p class="wpio-help-text">Questions? <a href="mailto:support@wpimg.io">support@wpimg.io</a></p>
                            </div>
                        <?php elseif ($status === 'paused' || $status === 'expired'): ?>
                            <div class="wpio-warning-banner wpio-warning-<?php echo esc_attr($status); ?>">
                                <p>
                                    <strong>Subscription <?php echo esc_html($status); ?>.</strong>
                                    Image optimization is currently disabled.
                                </p>
                                <p>
                                    <?php if ($status === 'expired'): ?>
                                        <a href="https://wpimg.io/billing" target="_blank">Renew your subscription</a>, then click Re-activate below.
                                        Questions? <a href="mailto:support@wpimg.io">support@wpimg.io</a>
                                    <?php else: ?>
                                        <a href="https://wpimg.io/billing" target="_blank">Resume your subscription</a>, then click Re-activate below.
                                        Questions? <a href="mailto:support@wpimg.io">support@wpimg.io</a>
                                    <?php endif; ?>
                                </p>
                            </div>

                            <div class="wpio-reactivate-form">
                                <div class="wpio-form-group">
                                    <label for="wpio-api-key">License Key</label>
                                    <input type="text" id="wpio-api-key" class="wpio-input" value="<?php echo esc_attr($api_key); ?>">
                                </div>
                                <button type="button" id="wpio-activate-btn" class="button button-primary">
                                    Re-activate
                                </button>
                                <div id="wpio-message" class="wpio-message hidden"></div>
                            </div>
                        <?php elseif ($status === 'service_discontinued'): ?>
                            <div class="wpio-warning-banner wpio-warning-service_discontinued">
                                <p>
                                    <strong>Service Discontinued.</strong>
                                    Image optimization has been automatically disabled to prevent broken images on your site.
                                </p>
                                <p>
                                    We apologize for any inconvenience. You can safely deactivate this plugin.
                                    Questions? <a href="mailto:support@wpimg.io">support@wpimg.io</a>
                                </p>
                            </div>
                        <?php else: ?>
                            <p class="wpio-success-msg">Images are being optimized!</p>
                            <p class="wpio-note">Note: Optimization only applies to visitors. Log out to see optimized images.</p>
                        <?php endif; ?>

                        <table class="wpio-info-table">
                            <tr>
                                <th>Domain</th>
                                <td><code><?php echo esc_html($current_domain); ?></code></td>
                            </tr>
                            <tr>
                                <th>CDN URL</th>
                                <td>
                                    <code><?php echo esc_html($cdn_url); ?></code>
                                    <p class="wpio-field-hint">Your images are automatically served through this address for faster loading worldwide.</p>
                                </td>
                            </tr>
                            <tr>
                                <th>Status</th>
                                <td>
                                    <span class="wpio-status-dot wpio-status-dot-<?php echo esc_attr($status); ?>"></span>
                                    <?php echo esc_html(ucfirst($status === 'past_due' ? 'past due' : $status)); ?>
                                </td>
                            </tr>
                        </table>

                        <!-- Stats Section -->
                        <div id="wpio-stats" class="wpio-stats">
                            <div class="wpio-stats-header">
                                <h3>Optimization Statistics</h3>
                                <span class="wpio-stats-period">All time</span>
                            </div>
                            <div class="wpio-stats-loading">Loading stats...</div>
                            <div class="wpio-stats-content hidden">
                                <div class="wpio-stats-cards">
                                    <div class="wpio-stat-card">
                                        <div class="wpio-stat-value" id="wpio-stat-images">-</div>
                                        <div class="wpio-stat-label">Images Optimized</div>
                                        <div class="wpio-stat-sub" id="wpio-stat-images-sub"></div>
                                    </div>
                                    <div class="wpio-stat-card">
                                        <div class="wpio-stat-value" id="wpio-stat-original">-</div>
                                        <div class="wpio-stat-label">Original Size</div>
                                        <div class="wpio-stat-sub" id="wpio-stat-original-avg">- avg</div>
                                    </div>
                                    <div class="wpio-stat-card">
                                        <div class="wpio-stat-value" id="wpio-stat-avif">-</div>
                                        <div class="wpio-stat-label">AVIF Size</div>
                                        <div class="wpio-stat-sub" id="wpio-stat-avif-avg">- avg</div>
                                    </div>
                                </div>
                                <div class="wpio-savings-bar-container">
                                    <div class="wpio-savings-header">
                                        <span>Bandwidth Saved</span>
                                        <span class="wpio-savings-percent" id="wpio-stat-percent">-%</span>
                                    </div>
                                    <div class="wpio-savings-bar">
                                        <div class="wpio-savings-fill" id="wpio-savings-fill"></div>
                                    </div>
                                    <div class="wpio-savings-detail" id="wpio-savings-detail">- saved</div>
                                </div>
                            </div>
                            <div class="wpio-stats-error hidden"></div>
                        </div>

                        <?php if ($is_free): ?>
                            <!-- Upgrade section for free users -->
                            <div class="wpio-upgrade-section">
                                <h3>Upgrade to Pro</h3>
                                <ul class="wpio-pro-benefits">
                                    <li>Unlimited bandwidth</li>
                                    <li>Custom CDN domain</li>
                                    <li>Priority support</li>
                                </ul>
                                <div class="wpio-form-group">
                                    <label for="wpio-upgrade-key">License Key</label>
                                    <input type="text" id="wpio-upgrade-key" class="wpio-input" placeholder="Enter your Pro license key">
                                </div>
                                <button type="button" id="wpio-upgrade-btn" class="button button-primary">
                                    Upgrade
                                </button>
                                <p class="wpio-help-text">
                                    Don't have a key? <a href="https://wpimg.io" target="_blank">Get one here</a>
                                </p>
                            </div>
                        <?php else: ?>
                            <p class="wpio-manage-link">
                                <a href="https://wpimg.io/billing" target="_blank">Manage subscription</a>
                            </p>
                        <?php endif; ?>

                        <details class="wpio-advanced">
                            <summary>Advanced</summary>
                            <div class="wpio-advanced-content">
                                <p class="wpio-advanced-desc">
                                    If images appear outdated after replacing files, clear the CDN cache.
                                    This is rarely needed—WordPress normally creates unique filenames.
                                </p>
                                <button type="button" id="wpio-purge-btn" class="button">
                                    Purge CDN Cache
                                </button>
                            </div>
                        </details>

                        <div id="wpio-message" class="wpio-message hidden"></div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Get the configured Site ID for the CDN rewriter
     */
    public static function get_site_id(): string {
        return get_option(self::OPTION_SITE_ID, '');
    }

    /**
     * Get the CDN URL (supports custom CNAME domains)
     */
    public static function get_cdn_url(): string {
        return get_option(self::OPTION_CDN_URL, '');
    }

    /**
     * Check if in development mode
     */
    public static function is_development(): bool {
        return get_option(self::OPTION_STATUS, '') === 'development';
    }

    /**
     * Check if subscription is expired or service discontinued (should disable rewriting)
     */
    public static function is_expired(): bool {
        $status = get_option(self::OPTION_STATUS, '');
        return in_array($status, ['expired', 'paused', 'service_discontinued'], true);
    }

    /**
     * Get the current cache version for URL cache busting
     */
    public static function get_cache_version(): string {
        return (string) get_option(self::OPTION_CACHE_VERSION, '1');
    }

    /**
     * Check subscription status via API (called by daily cron)
     */
    public static function check_subscription_status(): void {
        $api_key = get_option(self::OPTION_API_KEY, '');
        if (empty($api_key)) {
            return;
        }

        // Skip dev environments
        if (self::is_dev_environment()) {
            return;
        }

        $response = wp_remote_post(self::API_BASE . '/status', [
            'timeout' => 15,
            'headers' => ['Content-Type' => 'application/json'],
            'body' => json_encode(['api_key' => $api_key]),
        ]);

        if (is_wp_error($response)) {
            return; // Silent fail, will retry next day
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code === 200 && !empty($body['success']) && isset($body['status'])) {
            $old_status = get_option(self::OPTION_STATUS, '');
            $new_status = $body['status'];

            // Update status if changed
            if ($old_status !== $new_status) {
                update_option(self::OPTION_STATUS, $new_status);
            }
        }

        // Also fetch stats for usage tracking (for admin banners)
        self::update_usage_info($api_key);
    }

    /**
     * Fetch and store usage info for admin banners
     */
    private static function update_usage_info(string $api_key): void {
        $response = wp_remote_post(self::API_BASE . '/stats', [
            'timeout' => 15,
            'headers' => ['Content-Type' => 'application/json'],
            'body' => json_encode(['api_key' => $api_key]),
        ]);

        if (is_wp_error($response)) {
            return;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code === 200 && !empty($body['success'])) {
            if (isset($body['tier']) && $body['tier'] === 'free' && isset($body['limit'])) {
                set_transient('wpio_usage_info', [
                    'tier' => $body['tier'],
                    'limit' => $body['limit'],
                    'used' => $body['total_images'] ?? 0,
                ], DAY_IN_SECONDS);
            } else {
                delete_transient('wpio_usage_info');
            }
        }
    }
}
