<?php
/**
 * Plugin Updater
 *
 * Handles self-hosted plugin updates via WordPress native hooks.
 * Checks for updates from a static JSON file hosted on R2.
 */

if (!defined('ABSPATH')) {
    exit;
}

class WP_Image_Optimizer_Updater {

    /**
     * Plugin slug (directory name)
     */
    private const SLUG = 'wpimg-optimizer';

    /**
     * URL to the version metadata JSON file
     */
    private const METADATA_URL = 'https://plugin.wpimg.net/wpimg-optimizer.json';

    /**
     * Plugin basename (e.g., "wpimg-optimizer/wpimg-optimizer.php")
     * @var string
     */
    private $plugin_basename;

    /**
     * Current plugin version
     * @var string
     */
    private $current_version;

    /**
     * Initialize the updater
     *
     * @param string $plugin_file Main plugin file path (__FILE__ from main plugin)
     * @param string $version     Current plugin version
     */
    public function __construct(string $plugin_file, string $version) {
        $this->plugin_basename = plugin_basename($plugin_file);
        $this->current_version = $version;
    }

    /**
     * Register update hooks
     */
    public function init(): void {
        add_filter('pre_set_site_transient_update_plugins', [$this, 'check_for_update']);
        add_filter('plugins_api', [$this, 'plugin_info'], 10, 3);
    }

    /**
     * Check for plugin updates
     *
     * @param object $transient The update_plugins transient
     * @return object Modified transient
     */
    public function check_for_update($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }

        $remote_data = $this->get_remote_data();

        if ($remote_data === false) {
            return $transient;
        }

        if (version_compare($this->current_version, $remote_data->version, '<')) {
            $transient->response[$this->plugin_basename] = (object) [
                'slug'        => self::SLUG,
                'plugin'      => $this->plugin_basename,
                'new_version' => $remote_data->version,
                'url'         => $remote_data->author_profile ?? 'https://wpimg.io',
                'package'     => $remote_data->download_url,
                'icons'       => (array) ($remote_data->icons ?? []),
                'banners'     => (array) ($remote_data->banners ?? []),
                'tested'      => $remote_data->tested ?? '',
                'requires'    => $remote_data->requires ?? '',
                'requires_php'=> $remote_data->requires_php ?? '',
            ];
        } else {
            // No update available - add to no_update to prevent WordPress.org check
            $transient->no_update[$this->plugin_basename] = (object) [
                'slug'        => self::SLUG,
                'plugin'      => $this->plugin_basename,
                'new_version' => $this->current_version,
                'url'         => 'https://wpimg.io',
                'package'     => '',
            ];
        }

        return $transient;
    }

    /**
     * Provide plugin information for the "View details" popup
     *
     * @param false|object|array $result The result object
     * @param string             $action The type of information being requested
     * @param object             $args   Plugin arguments
     * @return false|object Plugin info or false
     */
    public function plugin_info($result, $action, $args) {
        if ($action !== 'plugin_information') {
            return $result;
        }

        if (!isset($args->slug) || $args->slug !== self::SLUG) {
            return $result;
        }

        $remote_data = $this->get_remote_data();

        // Fallback to bundled JSON if remote fetch fails
        if ($remote_data === false) {
            $remote_data = $this->get_bundled_data();
        }

        if ($remote_data === false) {
            return $result;
        }

        return (object) [
            'name'           => $remote_data->name,
            'slug'           => self::SLUG,
            'version'        => $remote_data->version,
            'author'         => '<a href="' . esc_url($remote_data->author_profile ?? 'https://wpimg.io') . '">' . esc_html($remote_data->author) . '</a>',
            'author_profile' => $remote_data->author_profile ?? 'https://wpimg.io',
            'requires'       => $remote_data->requires ?? '',
            'tested'         => $remote_data->tested ?? '',
            'requires_php'   => $remote_data->requires_php ?? '',
            'last_updated'   => $remote_data->last_updated ?? '',
            'sections'       => (array) ($remote_data->sections ?? []),
            'banners'        => (array) ($remote_data->banners ?? []),
            'icons'          => (array) ($remote_data->icons ?? []),
            'download_link'  => $remote_data->download_url,
        ];
    }

    /**
     * Fetch remote update data
     *
     * @return object|false Remote data object or false on failure
     */
    private function get_remote_data() {
        $response = wp_remote_get(self::METADATA_URL, [
            'timeout' => 10,
            'headers' => [
                'Accept' => 'application/json',
            ],
        ]);

        if (is_wp_error($response)) {
            return false;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            return false;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body);

        if (json_last_error() !== JSON_ERROR_NONE || empty($data->version)) {
            return false;
        }

        return $data;
    }

    /**
     * Read plugin metadata from bundled JSON file
     *
     * @return object|false Local data object or false on failure
     */
    private function get_bundled_data() {
        $json_file = dirname(dirname(__FILE__)) . '/wpimg-optimizer.json';

        if (!file_exists($json_file)) {
            return false;
        }

        $contents = file_get_contents($json_file);
        if ($contents === false) {
            return false;
        }

        $data = json_decode($contents);

        if (json_last_error() !== JSON_ERROR_NONE || empty($data->version)) {
            return false;
        }

        return $data;
    }
}
