<?php
/**
 * Plugin Name: Image Optimizer by wpimg.io
 * Description: Automatically optimizes images via wpimg.io service
 * Version: 1.0.20
 * Author: wpimg.io
 * Author URI: https://wpimg.io
 * License: GPL-2.0-or-later
 * Requires at least: 5.0
 * Tested up to: 6.9
 * Requires PHP: 7.1
 */

if (!defined('ABSPATH')) {
    exit;
}

define('WPIO_VERSION', '1.0.20');
define('WPIO_PLUGIN_FILE', __FILE__);
define('WPIO_DEBUG', defined('WP_DEBUG') && WP_DEBUG);

// Option names
define('WPIO_OPTION_API_KEY', 'wpio_api_key');
define('WPIO_OPTION_SITE_ID', 'wpio_site_id');
define('WPIO_OPTION_STATUS', 'wpio_status');

// Hook priorities - extreme values for CDN rewriting
// Early: start output buffer before other optimization plugins
// Late/Very Late: rewrite URLs after other plugins finish modifications
define('WPIO_PRIORITY_EARLY', PHP_INT_MIN + 100);
define('WPIO_PRIORITY_LATE', PHP_INT_MAX - 1000);
define('WPIO_PRIORITY_VERY_LATE', PHP_INT_MAX - 100);

function wpio_log($message) {
    if (defined('WPIO_DEBUG') && WPIO_DEBUG) {
        error_log('[wpimg.io] ' . $message);
    }
}

// Load classes
require_once __DIR__ . '/includes/class-settings.php';
require_once __DIR__ . '/includes/class-cdn-rewriter.php';
require_once __DIR__ . '/includes/class-updater.php';

/**
 * Check if current request is an AJAX or REST API request
 *
 * @return bool True if AJAX or REST context
 */
function wpio_is_ajax_or_rest(): bool {
    return wp_doing_ajax() || defined('REST_REQUEST');
}

/**
 * Check if current request is from an admin context
 *
 * Note: is_admin() returns true for ALL AJAX requests because they go through
 * wp-admin/admin-ajax.php which defines WP_ADMIN=true. So we check the referer
 * to determine if the request actually originated from an admin page.
 *
 * @return bool True if request originated from admin area
 */
function wpio_is_admin_context(): bool {
    // For non-AJAX/REST requests, is_admin() is reliable
    if (!wpio_is_ajax_or_rest()) {
        return is_admin();
    }

    // For AJAX/REST requests, check if the referer is from admin
    // wp_get_referer() checks both $_REQUEST['_wp_http_referer'] and $_SERVER['HTTP_REFERER']
    $referer = wp_get_referer();
    if ($referer) {
        // Check if referer contains /wp-admin/
        $admin_url = admin_url();
        return strpos($referer, $admin_url) !== false ||
               strpos($referer, '/wp-admin/') !== false;
    }

    // No referer available - assume frontend (safer for our use case:
    // better to serve CDN URLs than to break frontend lazy loading)
    return false;
}

/**
 * Get a cached CDN rewriter instance for the current request
 *
 * Returns null if plugin is not activated or subscription expired.
 * Uses static caching to avoid creating multiple instances per request.
 *
 * @return WP_Image_Optimizer_CDN_Rewriter|null Rewriter instance or null
 */
function wpio_get_rewriter(): ?WP_Image_Optimizer_CDN_Rewriter {
    static $rewriter = null;
    static $checked = false;

    // Only check once per request
    if ($checked) {
        return $rewriter;
    }
    $checked = true;

    $cdn_url = WP_Image_Optimizer_Settings::get_cdn_url();
    if (empty($cdn_url)) {
        return null;
    }

    if (WP_Image_Optimizer_Settings::is_expired()) {
        return null;
    }

    $api_key = get_option(WPIO_OPTION_API_KEY, '');
    $rewriter = new WP_Image_Optimizer_CDN_Rewriter($cdn_url, $api_key);

    return $rewriter;
}

/**
 * Activation hook - redirect to settings page and schedule cron
 */
register_activation_hook(__FILE__, function() {
    set_transient('wpio_activation_redirect', true, 30);

    // Schedule daily status check
    if (!wp_next_scheduled('wpio_daily_status_check')) {
        wp_schedule_event(time(), 'daily', 'wpio_daily_status_check');
    }
});

/**
 * Deactivation hook - clear scheduled cron
 */
register_deactivation_hook(__FILE__, function() {
    wp_clear_scheduled_hook('wpio_daily_status_check');
});

/**
 * Redirect to settings after activation
 */
add_action('admin_init', function() {
    if (get_transient('wpio_activation_redirect')) {
        delete_transient('wpio_activation_redirect');

        // Don't redirect on bulk activate
        if (isset($_GET['activate-multi'])) {
            return;
        }

        wp_safe_redirect(admin_url('options-general.php?page=wpimg-optimizer'));
        exit;
    }
});

/**
 * Initialize admin settings and updater
 */
add_action('plugins_loaded', function() {
    if (is_admin()) {
        $settings = new WP_Image_Optimizer_Settings();
        $settings->init();
    }

    // Initialize updater (hooks only fire in admin context)
    $updater = new WP_Image_Optimizer_Updater(WPIO_PLUGIN_FILE, WPIO_VERSION);
    $updater->init();
});

/**
 * Daily cron: check subscription status via API
 */
add_action('wpio_daily_status_check', function() {
    WP_Image_Optimizer_Settings::check_subscription_status();
});

/**
 * Add settings link to plugins list
 */
add_filter('plugin_action_links_' . plugin_basename(__FILE__), function($links) {
    $settings_link = '<a href="' . admin_url('options-general.php?page=wpimg-optimizer') . '">Settings</a>';
    array_unshift($links, $settings_link);
    return $links;
});

/**
 * Inject preconnect and fallback script into <head>
 */
add_action('wp_head', function() {
    // Skip if not activated
    $cdn_url = WP_Image_Optimizer_Settings::get_cdn_url();
    if (empty($cdn_url)) {
        return;
    }

    // Skip in admin, preview, customizer
    if (is_admin() || is_preview() || is_customize_preview()) {
        return;
    }

    // Skip if subscription expired
    if (WP_Image_Optimizer_Settings::is_expired()) {
        return;
    }

    // Extract domain from CDN URL for fallback script
    $cdn_domain = parse_url($cdn_url, PHP_URL_HOST) ?: '';

    // Preconnect for faster CDN connection
    echo '<link rel="preconnect" href="' . esc_url($cdn_url) . '">' . "\n";

    // Fallback script: loads original image if CDN fails
    // CDN URL format: /{version}.{sig}{path} - extract path and serve from origin
    // No tracking needed: after fallback, src no longer contains CDN domain
    ?>
<script>
(function(){
var F='__wpio',D='<?php echo esc_js($cdn_domain); ?>';
document.addEventListener('error',function(e){
var el=e.target;
if(!(el instanceof HTMLImageElement)||el[F])return;
var src=el.currentSrc||el.src||'';
if(src.indexOf(D)===-1)return;
el[F]=1;
try{
var m=new URL(src).pathname.match(/^\/[^\/]+\.[^\/]+(\/.*)/);
if(!m)return;
var path=m[1],origin='<?php echo esc_js(site_url()); ?>';
if(el.srcset)el.srcset='';
if(el.dataset.srcset)el.dataset.srcset='';
var pic=el.closest('picture');
if(pic){var s=pic.querySelectorAll('source');for(var i=0;i<s.length;i++)s[i].srcset='';}
el.src=origin+path;
}catch(x){}
},true);
})();
</script>
<?php
}, 1); // Priority 1: very early in <head>

/**
 * Start output buffering for CDN rewriting
 */
add_action('template_redirect', function() {
    wpio_log('template_redirect fired');

    // Skip in admin, preview, or customizer
    if (is_admin() || is_preview() || is_customize_preview()) {
        wpio_log('Skipping: admin/preview/customizer');
        return;
    }

    // Skip non-HTML contexts
    if (is_feed() || is_robots() || wp_doing_cron()) {
        wpio_log('Skipping: feed/robots/cron');
        return;
    }
    if (defined('XMLRPC_REQUEST') || defined('REST_REQUEST') || defined('WP_CLI')) {
        wpio_log('Skipping: XMLRPC/REST/CLI');
        return;
    }

    // Skip if not activated (no cdn_url configured)
    $cdn_url = WP_Image_Optimizer_Settings::get_cdn_url();
    wpio_log('CDN URL: ' . ($cdn_url ?: '(empty)'));
    if (empty($cdn_url)) {
        wpio_log('Skipping: no cdn_url');
        return;
    }

    // Skip if subscription expired
    if (WP_Image_Optimizer_Settings::is_expired()) {
        wpio_log('Skipping: subscription expired');
        return;
    }

    // Skip for AJAX requests (handled by the_content filter)
    if (wp_doing_ajax()) {
        wpio_log('Skipping: AJAX request');
        return;
    }

    // Skip when admin bar is showing (logged-in admins viewing frontend)
    if (function_exists('is_admin_bar_showing') && is_admin_bar_showing()) {
        wpio_log('Skipping: admin bar showing');
        return;
    }

    wpio_log('Starting output buffering, current ob level: ' . ob_get_level());
    $api_key = get_option(WPIO_OPTION_API_KEY, '');
    $rewriter = new WP_Image_Optimizer_CDN_Rewriter($cdn_url, $api_key);
    $result = ob_start([$rewriter, 'process']);
    wpio_log('ob_start result: ' . ($result ? 'true' : 'false') . ', new ob level: ' . ob_get_level());
}, WPIO_PRIORITY_EARLY);

// Note: No shutdown hook needed - PHP automatically flushes all output buffers
// at script termination, and WordPress's wp_ob_end_flush_all() (priority 1)
// handles this as well. Our ob_start callback is invoked automatically.

/**
 * Fallback filter for AJAX/REST partial content updates
 *
 * Only applies to frontend AJAX/REST requests (e.g., infinite scroll, lazy loading).
 * Admin-side AJAX is skipped to avoid conflicts with page builders, media library,
 * and other plugins that may expect local URLs for image manipulation.
 */
add_filter('the_content', function($content) {
    // Skip admin-side AJAX (media library, page builders, block editor)
    if (wpio_is_admin_context()) {
        return $content;
    }

    if (!wpio_is_ajax_or_rest()) {
        return $content;
    }

    $rewriter = wpio_get_rewriter();
    if (!$rewriter) {
        return $content;
    }

    return $rewriter->process($content);
}, WPIO_PRIORITY_VERY_LATE);

/**
 * Rewrite attachment URLs in AJAX/REST contexts
 *
 * These hooks catch images loaded via WordPress functions like:
 * - the_post_thumbnail() / get_the_post_thumbnail()
 * - wp_get_attachment_image()
 * - wp_get_attachment_url()
 *
 * This is needed because output buffering is skipped for AJAX/REST,
 * and the_content filter only catches post content, not featured images
 * or programmatically inserted images.
 *
 * Only applies to frontend AJAX/REST requests. Admin-side AJAX is skipped
 * to avoid conflicts with plugins that expect local URLs for processing.
 */
add_filter('wp_get_attachment_url', function($url) {
    // Skip admin-side AJAX (media library, page builders, block editor)
    if (wpio_is_admin_context()) {
        return $url;
    }

    if (!wpio_is_ajax_or_rest()) {
        return $url;
    }

    $rewriter = wpio_get_rewriter();
    if (!$rewriter) {
        return $url;
    }

    return $rewriter->rewrite_url($url);
}, WPIO_PRIORITY_LATE);

/**
 * Rewrite srcset URLs in AJAX/REST contexts
 *
 * The srcset attribute contains multiple image URLs for responsive images.
 * Format: $sources[width] = ['url' => '...', 'descriptor' => 'w', 'value' => width]
 *
 * Only applies to frontend AJAX/REST requests. Admin-side AJAX is skipped
 * to avoid conflicts with plugins that expect local URLs for processing.
 */
add_filter('wp_calculate_image_srcset', function($sources) {
    // Skip admin-side AJAX (media library, page builders, block editor)
    if (wpio_is_admin_context()) {
        return $sources;
    }

    if (!wpio_is_ajax_or_rest()) {
        return $sources;
    }

    if (!is_array($sources)) {
        return $sources;
    }

    $rewriter = wpio_get_rewriter();
    if (!$rewriter) {
        return $sources;
    }

    foreach ($sources as $width => $source) {
        if (isset($source['url'])) {
            $sources[$width]['url'] = $rewriter->rewrite_url($source['url']);
        }
    }

    return $sources;
}, WPIO_PRIORITY_LATE);
